"""Stefan Spence 25.05.20
Create a simulation of Raman Sideband Cooling using qutip and following
Adam Kaufman's thesis.
"""
#import packages
import matplotlib.pyplot as plt
import numpy as np
from qutip import *
from scipy.interpolate import interp1d
from scipy.signal import tukey

# Constants
hbar = 1.0545718e-34 # reduced Planck's constant in m^2 kg / s
kB = 1.38064852e-23  # Boltzmann's constant in m^2 kg s^-2 K^-1

def make_hamiltonians(Np=15, W=150*2*np.pi, D=150*2*np.pi, Temp=15, LD=(2.07/150)**0.5, LDop=(2.07/150)**0.5, 
        Rabi=30*2*np.pi, Gamma=100, Nop=3, Depump=1):
    """Produce the bare Jaynes-Cummings Hamiltonian, the Raman beam 
    interaction, and the optical pumping dissipation operator.
    Np = 15        # Hilbert space dimension for the external motion    
    W = 150*2*np.pi  # Harmonic oscillator frequency (2pi kHz)
    D = 0.977 * w  # detuning from 2-photon resonance in (2pi kHz)
    Temp = 15           # Temperature (uK)
    wr = 2.07*2*np.pi # Recoil frequency (2pi kHz) -- 2.07 Cs, 3.77 Rb
    Rabi = 31*2*np.pi  # Rabi frequency for the carrier transition of the raman beams (2pi kHz)
    Gamma = 100      # Rate of optical pumping (rad / ms)
    Depump = gamma/100 # Rate of depumping - OP in opposite direction (rad/ms)
    Nop = 3          # Number of OP + Repump photons it takes to pump the atoms back to g2
    LD = np.sqrt(2*wr/w) # Lamb-dicke parameter for raman beams (radial direction)
    LDop = np.sqrt(wr/w) # Lamb-dicke parameter for the optical pumping beam"""
    # phonon operators
    a = tensor(destroy(Np),qeye(2)) #anhilation operator
    #atomic operators here, #g1->0,g2->1
    sm_g2g1 = tensor(qeye(Np), sigmap()) # spin flip g2->g1
    #Hamiltonians
    Hb = -D / 2 * tensor(qeye(Np), sigmaz()) # bare Hamiltonian, internal
    for i in range(Np): Hb += W*(i)*tensor(fock_dm(Np,i), qeye(2)) # number states
    Hr = Rabi/2 * sm_g2g1 * (-1j*LD*(a + a.dag())).expm() # Raman Hamiltonian
    # Dissipation operators for decoherence
    Hop = np.sqrt(Gamma) * sm_g2g1 * (1j*LDop*(a+a.dag())).expm()**Nop #Optical pumping operator
    Hdp = np.sqrt(Depump) * tensor(qeye(Np), sigmam()) * (1j*LDop*(a+a.dag())).expm()**Nop #Optical pumping operator
    return Hb, Hr+Hr.dag(), Hop, Hdp

def simulateDP(Np=20, W=140*2*np.pi, D=140*2*np.pi, Temp=15, LD=0.172, LDop=0.121, Rabi=30*2*np.pi, Ntp=3000,
        Gamma=100, Nop=3, N=50, dR=0.5, dOP=1, psi0=None, Depump=1, pgb=ui.EnhancedTextProgressBar()):
    """Run a simulation of Raman sideband cooling with the given parameters, including depump from bad polarisation."""
    if psi0 == None:
        n_th = n_thermal(W, Temp*1e-9*kB/hbar) # average phonon number due to temperature
        psi0 = tensor(thermal_dm(Np, n_th, method='operator'), fock_dm(2,0)) #thermal state
    tOP = 2*np.pi/(Gamma) #OP pulse duration, ms
    tR = np.pi/(Rabi*LD)   # Maximum Raman pulse duration, ms
    Hb, Hr, Hop, Hdp = make_hamiltonians(Np=Np, W=W, D=D, Temp=Temp, LD=LD, LDop=LDop,
        Rabi=Rabi, Gamma=Gamma, Nop=Nop, Depump=Depump)
    tlist = np.linspace(0, (dR*tR + dOP*tOP)*N, Ntp)
    Hop_coeff = (tlist%(dR*tR + dOP*tOP) > dR*tR).astype(float)
    Hr_coeff = (tlist%(dR*tR + dOP*tOP) < dR*tR).astype(float) 
    return tlist, mesolve([Hb, [Hr, Hr_coeff]], psi0, tlist, [[Hop, Hop_coeff], [Hdp, Hop_coeff]], progress_bar=pgb)

def simulate(Np=20, W=140*2*np.pi, D=140*2*np.pi, Temp=15, LD=0.172, LDop=0.121, Rabi=30*2*np.pi, Ntp=3000,
        Gamma=100, Nop=3, N=50, dR=0.5, dOP=1, psi0=None, Depump=1,pgb=ui.EnhancedTextProgressBar()):
    """Run a simulation of Raman sideband cooling with the given parameters. Exclude depump"""
    if psi0 == None:
        n_th = n_thermal(W, Temp*1e-9*kB/hbar) # average phonon number due to temperature
        psi0 = tensor(thermal_dm(Np, n_th, method='operator'), fock_dm(2,0)) #thermal state
    tOP = 2*np.pi/(Gamma) #OP pulse duration, ms
    tR = np.pi/(Rabi*LD)   # Maximum Raman pulse duration, ms
    Hb, Hr, Hop, Hdp = make_hamiltonians(Np=Np, W=W, D=D, Temp=Temp, LD=LD, LDop=LDop,
        Rabi=Rabi, Gamma=Gamma, Nop=Nop, Depump=Depump)
    tlist = np.linspace(0, (dR*tR + dOP*tOP)*N, Ntp)
    Hop_coeff = (tlist%(dR*tR + dOP*tOP) > dR*tR).astype(float)
    Hr_coeff = (tlist%(dR*tR + dOP*tOP) < dR*tR).astype(float) 
    return tlist, mesolve([Hb, [Hr, Hr_coeff]], psi0, tlist, [[Hop, Hop_coeff]], progress_bar=pgb)

def simulate_mc(Np=20, W=140*2*np.pi, D=140*2*np.pi, Temp=15, LD=0.172, LDop=0.121, Rabi=30*2*np.pi, Ntp=3000,
        Gamma=100, Nop=3, Nt=300, N=50, dR=1, dOP=1, psi0=None, Depump=1):
    """Run a Monte Carlo simulation of RSC with the given parameters"""
    if psi0 == None:
        n_th = n_thermal(W, Temp*2e-9*kB/hbar) # extra factor of 2 for MC method???
        th = np.diag(thermal_dm(Np, n_th, method='operator')) #thermal state
        psi0 = sum(tensor(fock(Np,i)*th[i], fock(2,0)) for i in range(Np)) # as state vector
    tOP = 2*np.pi/(Gamma) #OP pulse duration, ms
    tR = np.pi/(Rabi*LD)   # Maximum Raman pulse duration, ms
    Hb, Hr, Hop, Hdp = make_hamiltonians(Np=Np, W=W, D=D, Temp=Temp, LD=LD, LDop=LDop,
        Rabi=Rabi, Gamma=Gamma, Nop=Nop, Depump=1)
    tlist = np.linspace(0,(dR*tR + dOP*tOP)*N,Ntp) # time points for the simulation
    #Define the time dependent coefficients for all the hamiltonians as an array
    Hr_coeff = (tlist%(dR*tR + dOP*tOP) < dR*tR).astype(float)
    return tlist, mcsolve([Hb, [Hr, Hr_coeff]], psi0, tlist, [[Hop, 1-Hr_coeff], [Hdp, 1-Hr_coeff]], progress_bar=ui.EnhancedTextProgressBar())


def simulate_Gauss(Np=20, W=140*2*np.pi, D=140*2*np.pi, Temp=15, LD=0.172, LDop=0.121, Rabi=30*2*np.pi, Ntp=1500,
        Gamma=100, Nop=3, N=50, dR=1, dOP=1, psi0=None, Depump=0.1, pw=3, pgb=ui.EnhancedTextProgressBar()):
    """Run a simulation of Raman sideband cooling with the given parameters. 
    Raman pulses have a Gaussian pulse shape. pw is the # s.d. of the Gaussian to include in the pulse."""
    if psi0 == None:
        n_th = n_thermal(W, Temp*1e-9*kB/hbar) # average phonon number due to temperature
        psi0 = tensor(thermal_dm(Np, n_th, method='operator'), fock_dm(2,0)) #thermal state
    tOP = 2*np.pi/(Gamma) #OP pulse duration, ms
    tR = np.pi/(Rabi*LD)   # Maximum Raman pulse duration, ms
    pw *= np.sqrt(2/np.pi) # convert to std devs
    Hb, Hr, Hop, Hdp = make_hamiltonians(Np=Np, W=W, D=D, Temp=Temp, LD=LD, LDop=LDop,
        Rabi=Rabi, Gamma=Gamma, Nop=Nop, Depump=Depump)
    tlist = np.linspace(0,(dR*tR*pw + dOP*tOP)*N,Ntp) # time points for the simulation
    Hop_coeff = (tlist%(dR*tR*pw + dOP*tOP) > dR*tR*pw).astype(float)
    Hr_coeff = (tlist%(dR*tR*pw + dOP*tOP) < dR*tR*pw).astype(float) * np.exp(-(tlist%(dR*tR*pw + dOP*tOP)-dR*tR*pw/2)**2*np.pi/(dR*tR)**2)
    return tlist, mesolve([Hb, [Hr, Hr_coeff]], psi0, tlist, [[Hop, Hop_coeff], [Hdp, Hop_coeff]], progress_bar=pgb)

def simulate_BlckHarris(Np=20, W=140*2*np.pi, D=140*2*np.pi, Temp=15, LD=0.172, LDop=0.121, Rabi=30*2*np.pi, Ntp=3000,
        Gamma=100, Nop=3, N=50, dR=1, dOP=1, psi0=None, Depump=0.1, pgb=ui.EnhancedTextProgressBar()):
    """Run a simulation of Raman sideband cooling with the given parameters. 
    Raman pulses have a Blackman-Harris pulse shape."""
    if psi0 == None:
        n_th = n_thermal(W, Temp*1e-9*kB/hbar) # average phonon number due to temperature
        psi0 = tensor(thermal_dm(Np, n_th, method='operator'), fock_dm(2,0)) #thermal state
    tOP = 2*np.pi/(Gamma) #OP pulse duration, ms
    tBl = dR * np.pi/(Rabi*LD) /0.35875 # Blackman pulse duration, ms
    Hb, Hr, Hop, Hdp = make_hamiltonians(Np=Np, W=W, D=D, Temp=Temp, LD=LD, LDop=LDop,
        Rabi=Rabi, Gamma=Gamma, Nop=Nop, Depump=Depump)
    tlist = np.linspace(0,(tBl + dOP*tOP)*N,Ntp) # time points for the simulation
    tperiod = tlist%(tBl + dOP*tOP) # periodic time of single pulse sequence
    Hop_coeff = (tperiod > tBl).astype(float)
    tscaled = tperiod / tBl # Blackman-Harris has zeros at t=0,1
    Hr_coeff = (0.35875 - 0.48829*np.cos(2*np.pi*tscaled) +0.14128*np.cos(4*np.pi*tscaled) - 0.01168*np.cos(6*np.pi*tscaled)) * (1-Hop_coeff)
    return tlist, mesolve([Hb, [Hr, Hr_coeff]], psi0, tlist, [[Hop, Hop_coeff], [Hdp, Hop_coeff]], progress_bar=pgb)

def simulate_Tukey(Np=20, W=140*2*np.pi, D=140*2*np.pi, Temp=15, LD=0.172, LDop=0.121, Rabi=30*2*np.pi, Ntp=100,
        Gamma=100, Nop=3, N=50, dR=1, dOP=1, a=2/3., psi0=None, Depump=0.1, pgb=ui.EnhancedTextProgressBar()):
    """Run a simulation of Raman sideband cooling with the given parameters. 
    Raman pulses have a Tukey / tapered cosine pulse shape."""
    if psi0 == None:
        n_th = n_thermal(W, Temp*1e-9*kB/hbar) # average phonon number due to temperature
        psi0 = tensor(thermal_dm(Np, n_th, method='operator'), fock_dm(2,0)) #thermal state
    tOP = 2*np.pi/(Gamma) #OP pulse duration, ms
    tT = dR * np.pi/(Rabi*LD) /(1-a/2) # tapered-cosine pulse duration, ms
    Hb, Hr, Hop, Hdp = make_hamiltonians(Np=Np, W=W, D=D, Temp=Temp, LD=LD, LDop=LDop,
        Rabi=Rabi, Gamma=Gamma, Nop=Nop, Depump=Depump)
    tlist = np.linspace(0,(tT + dOP*tOP)*N,Ntp*N) # time points for the simulation
    Hop_coeff = (tlist%(tT + dOP*tOP) > tT).astype(float)
    ind0 = int(np.around(Ntp * tT / (tT + dOP*tOP))) # number of time steps for Raman pulse
    ind1 = int(np.around(Ntp * dOP*tOP / (tT + dOP*tOP))) # number of time steps for OP pulse
    Hr_coeff = np.concatenate([tukey(ind0), np.zeros(ind1)]*N)
    return tlist, mesolve([Hb, [Hr, Hr_coeff]], psi0, tlist, [[Hop, Hop_coeff], [Hdp, Hop_coeff]], progress_bar=pgb)

def sim_Tukey_Scat(Np=20, W=140*2*np.pi, D=140*2*np.pi, Temp=15, LD=0.172, LDop=0.121, Rabi=30*2*np.pi, Ntp=100,
        Gamma=100, Nop=3, N=50, dR=1, dOP=1, a=2/3., psi0=None, Depump=0.1, Scat=0.1, pgb=ui.EnhancedTextProgressBar()):
    """Run a simulation of Raman sideband cooling with the given parameters. 
    Raman pulses have a Tukey pulse shape. Account for polarisation impurity and off-resonant scattering."""
    if psi0 == None:
        n_th = n_thermal(W, Temp*1e-9*kB/hbar) # average phonon number due to temperature
        psi0 = tensor(thermal_dm(Np, n_th, method='operator'), fock_dm(2,0)) #thermal state
    tOP = 2*np.pi/(Gamma) #OP pulse duration, ms
    tT = dR * np.pi/(Rabi*LD) /(1-a/2) # tapered-cosine pulse duration, ms
    Hb, Hr, Hop, Hdp = make_hamiltonians(Np=Np, W=W, D=D, Temp=Temp, LD=LD, LDop=LDop,
        Rabi=Rabi, Gamma=Gamma, Nop=Nop, Depump=Depump)
    Hscat = Scat * (Hop/Gamma + Hdp/Depump)
    tlist = np.linspace(0,(tT + dOP*tOP)*N,Ntp*N) # time points for the simulation
    Hop_coeff = (tlist%(tT + dOP*tOP) > tT).astype(float)
    ind0 = int(np.around(Ntp * tT / (tT + dOP*tOP))) # number of time steps for Raman pulse
    ind1 = int(np.around(Ntp * dOP*tOP / (tT + dOP*tOP))) # number of time steps for OP pulse
    Hr_coeff = np.concatenate([tukey(ind0), np.zeros(ind1)]*N)
    return tlist, mesolve([Hb, [Hr, Hr_coeff]], psi0, tlist, [Hscat, [Hop, Hop_coeff], [Hdp, Hop_coeff]], progress_bar=pgb)


if __name__ == "__main__":
    n = tensor(num(10), qeye(2)) # number operator
    tlist, result = simulateDP(Np=10, Ntp=1500, dR=1, N=30)
    fig1, ax1 = plt.subplots(figsize=(10,6))
    ax1.plot(tlist, np.real(expect(n, result.states)))
    ax1.set_xlabel('Time (ms)')
    ax1.set_ylabel('Average phonon number')
    ax1.set_yscale('log')
    ax1.set_title('Full exponential matrices for $H_R$ and $H_{OP}$')
    plt.show()
