#!/bin/bash

# VASP to CIF Converter - Bash Script
# This script finds VASP files and opens them in VESTA for manual conversion to CIF

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Function to check if VESTA is available
check_vesta() {
    if ! command -v open &> /dev/null; then
        print_error "This script is designed for macOS (requires 'open' command)"
        exit 1
    fi
    
    if ! ls /Applications/VESTA.app &> /dev/null; then
        print_error "VESTA.app not found in /Applications/"
        print_info "Please install VESTA or check the installation path"
        exit 1
    fi
    
    print_success "VESTA found in /Applications/"
}

# Function to find VASP files
find_vasp_files() {
    local search_dir="$1"
    local files=()
    
    print_info "Searching for VASP files in: $search_dir"
    
    # Find files with common VASP extensions and names
    while IFS= read -r -d '' file; do
        files+=("$file")
    done < <(find "$search_dir" -type f \( -name "*.vasp" -o -name "*.VASP" -o -name "CONTCAR*" -o -name "POSCAR*" \) -print0)
    
    echo "${files[@]}"
}

# Function to get CIF filename from VASP filename
get_cif_filename() {
    local vasp_file="$1"
    local dir=$(dirname "$vasp_file")
    local basename=$(basename "$vasp_file")
    
    # Remove extension if it exists, then add .cif
    local name_no_ext="${basename%.*}"
    # If no extension was removed, use the full name
    if [[ "$name_no_ext" == "$basename" ]]; then
        echo "$dir/${basename}.cif"
    else
        echo "$dir/${name_no_ext}.cif"
    fi
}

# Function to process a single file
process_file() {
    local vasp_file="$1"
    local cif_file="$2"
    
    print_info "Processing: $(basename "$vasp_file")"
    print_info "Expected CIF output: $(basename "$cif_file")"
    
    # Open the file in VESTA
    open -a VESTA "$vasp_file"
    
    # Wait a moment for VESTA to load
    sleep 2
    
    echo ""
    echo "======================================================"
    echo "VESTA should now be open with: $(basename "$vasp_file")"
    echo "======================================================"
    echo "Please follow these steps:"
    echo "1. Go to File → Export Data..."
    echo "2. Select 'CIF' from the format dropdown"
    echo "3. Save as: $(basename "$cif_file")"
    echo "4. Save in directory: $(dirname "$cif_file")"
    echo "======================================================"
    echo ""
    
    # Wait for user confirmation
    while true; do
        read -p "Press [Enter] when export is complete, [s] to skip, or [q] to quit: " choice
        case $choice in
            "" )
                # Check if CIF file was created
                if [[ -f "$cif_file" ]]; then
                    print_success "CIF file created: $(basename "$cif_file")"
                    return 0
                else
                    print_warning "CIF file not found. Please check the export."
                    read -p "Continue anyway? [y/N]: " continue_choice
                    if [[ $continue_choice =~ ^[Yy]$ ]]; then
                        return 0
                    fi
                fi
                ;;
            [Ss]* )
                print_warning "Skipped: $(basename "$vasp_file")"
                return 1
                ;;
            [Qq]* )
                print_info "Quitting..."
                exit 0
                ;;
            * )
                echo "Please press Enter, 's' to skip, or 'q' to quit."
                ;;
        esac
    done
}

# Function to show usage
show_usage() {
    echo "Usage: $0 [directory]"
    echo ""
    echo "Convert VASP files to CIF format using VESTA"
    echo ""
    echo "Arguments:"
    echo "  directory    Directory to search for VASP files (default: current directory)"
    echo ""
    echo "Options:"
    echo "  -h, --help   Show this help message"
    echo "  -d, --dry-run Show files that would be processed without processing them"
    echo ""
    echo "Examples:"
    echo "  $0                    # Process current directory"
    echo "  $0 /path/to/files     # Process specific directory"
    echo "  $0 -d                 # Dry run on current directory"
    echo ""
}

# Function for dry run
dry_run() {
    local search_dir="$1"
    local files=($(find_vasp_files "$search_dir"))
    
    if [[ ${#files[@]} -eq 0 ]]; then
        print_warning "No VASP files found in $search_dir"
        return
    fi
    
    print_info "Found ${#files[@]} VASP files:"
    echo ""
    
    for file in "${files[@]}"; do
        local cif_file=$(get_cif_filename "$file")
        echo "  VASP: $file"
        echo "  CIF:  $cif_file"
        echo ""
    done
}

# Main function
main() {
    local search_dir="."
    local dry_run_mode=false
    
    # Parse command line arguments
    while [[ $# -gt 0 ]]; do
        case $1 in
            -h|--help)
                show_usage
                exit 0
                ;;
            -d|--dry-run)
                dry_run_mode=true
                shift
                ;;
            -*)
                print_error "Unknown option: $1"
                show_usage
                exit 1
                ;;
            *)
                search_dir="$1"
                shift
                ;;
        esac
    done
    
    # Check if directory exists
    if [[ ! -d "$search_dir" ]]; then
        print_error "Directory not found: $search_dir"
        exit 1
    fi
    
    # Convert to absolute path
    search_dir=$(cd "$search_dir" && pwd)
    
    print_info "VASP to CIF Converter"
    print_info "Search directory: $search_dir"
    
    # Check VESTA availability
    check_vesta
    
    # Find VASP files
    local files=($(find_vasp_files "$search_dir"))
    
    if [[ ${#files[@]} -eq 0 ]]; then
        print_warning "No VASP files found in $search_dir"
        exit 0
    fi
    
    print_info "Found ${#files[@]} VASP files"
    
    # Dry run mode
    if [[ $dry_run_mode == true ]]; then
        dry_run "$search_dir"
        exit 0
    fi
    
    # Process files
    local successful=0
    local skipped=0
    local total=${#files[@]}
    
    echo ""
    print_info "Starting conversion process..."
    echo ""
    
    for file in "${files[@]}"; do
        local cif_file=$(get_cif_filename "$file")
        
        if process_file "$file" "$cif_file"; then
            ((successful++))
        else
            ((skipped++))
        fi
        
        echo ""
        print_info "Progress: $((successful + skipped))/$total files processed"
        echo ""
    done
    
    # Summary
    echo "======================================================"
    echo "CONVERSION SUMMARY"
    echo "======================================================"
    echo "Total files found: $total"
    echo "Successfully processed: $successful"
    echo "Skipped: $skipped"
    echo "======================================================"
    
    if [[ $successful -gt 0 ]]; then
        print_success "Conversion process completed!"
    else
        print_warning "No files were successfully converted."
    fi
}

# Run main function with all arguments
main "$@"
