# Code to characterize H-pivot-minor-free graphs in terms of minimal forbidden induced subgraphs.
# Copyright (C) 2018 Konrad K. Dabrowski
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

# To be used with Sage version 8.2 or higher.
# Before using, install sortedcontainers using
#    ./sage -pip install sortedcontainers
#
# To calculate the forbidden induced subgraph characterization for
# H-pivot-minor-free graphs, set input_graph to H (some examples are given below) and run
#    ./sage wg2018.sage

# input_graph is the graph H for which we are classifying H-pivot-minor-free graphs.
# We assume that input_graph contains at least two vertices.
input_graph=graphs.ClawGraph()
#input_graph=graphs.PathGraph(4)
#input_graph=2*graphs.PathGraph(2)
#input_graph=2*graphs.PathGraph(1)+graphs.PathGraph(2)

# If this set to True, the minimal forbidden induced subgraphs are output in graph6 format
show_graph6=False

# If this set to True, the minimal forbidden induced subgraphs are drawn
# To use this, the code should be run in a Sage notebook, not from the commandline.
draw_graphs=False

from sortedcontainers import SortedSet

def local_complementation(G, v):
    for (i,j) in sage.combinat.subset.Subsets_sk(G.neighbors(v),2):
        if G.has_edge(i,j):
            G.delete_edge(i,j)
        else:
            G.add_edge(i,j)

def edge_pivot(G, v, w):
    if not G.has_edge(v,w):
        raise ValueError('Attempt to pivot on a non-edge.')
    local_complementation(G,v)
    local_complementation(G,w)
    local_complementation(G,v)

def contains_forb_graph(G):
    for i in range(0,k):
        for H in F[i]:
            if G.subgraph_search(H, induced=True) is not None:
                return true
    return false

# Throughout this code, graphs will always be stored with canonical labelling.
# This means that if two graphs F,G are isomorphic then F==G will evaluate to True.
# To make SortedSet work correctly, graphs must be stored in a way that is hashable.
# To do this, graphs added to sets must have immutable set to True.
input_graph=input_graph.canonical_label().copy(immutable=True)
k=input_graph.order()

# We will now close the set {input_graph} under edge pivots.
# We start with unprocessed_graphs={input_graph} and processed_graphs={}.
# Then, for each graph G in unprocessed_graphs, we:
# 1. Move G to processed_graphs
# 2. For each graph tmp that can be obtained from G by doing one edge pivot, if tmp is not in processed_graphs, then we add tmp to unprocessed_graphs.
# Note that unprocessed_graphs is a set, so any duplicates will be automatically removed.
unprocessed_graphs=SortedSet([input_graph])
processed_graphs=SortedSet()
while unprocessed_graphs:
    G=unprocessed_graphs.pop()
    processed_graphs.add(G)
    for (i,j,z) in G.edges(): # Sage describes edges as (first_vertex,second_vertex,label), so z will always be set to None and can be safely ignored.
        tmp=G.copy(immutable=False)
        edge_pivot(tmp,i,j)
        tmp=tmp.canonical_label()
        tmp=tmp.copy(immutable=True)
        if tmp not in processed_graphs:
            unprocessed_graphs.add(tmp)

# F[i] will contain the minimal forbidden induced subgraphs on i vertices.
# We start by filling in the trivial ones.
F=[]
for i in range(0,k):
    F.append(SortedSet())
F.append(processed_graphs)

# This will fail if k<=1, which only happens if the input graph is a 1 or 0-vertex graph
while F[k] or F[k-1]:
    k=k+1
    F.append(SortedSet())
    # For each graph G in F[k-1], we try every possible way to add a vertex v, then pivot on an edge incident to that vertex.
    for G in F[k-1]:
        for S in Subsets(G.vertices()): # S is the set of vertices that will be adjacent to v
            if len(S)>0: # Can assume that the new vertex has at least one neighbour
                tmp=G.copy(immutable=False)
                tmp.add_vertex(k-1) # The vertices of graphs in F[k-1] have labels 0,...,k-2 due to storing the graphs with canonical labelling
                for i in S:
                    tmp.add_edge(i,k-1)
                for i in S:
                    tmp2=tmp.copy()
                    edge_pivot(tmp2,i,k-1)
                    if not contains_forb_graph(tmp2):
                        F[k].add(tmp2.canonical_label().copy(immutable=True))
    # For each graph G in F[k-2], we try every possible way to add adjacent vertices u,b, then pivot on uv.
    for G in F[k-2]:
        vertex_sets=[]
        for i in Subsets(G.vertices()):
            # Can assume that each of the new vertices has at least one neighbour, otherwise pivoting uv does nothing.
            if len(i)>0:
                vertex_sets.append(i)
        # Can assume that the two new vertices have different neighbourhoods, otherwise pivoting uv does nothing.
        for (S,T) in sage.combinat.subset.Subsets_sk(vertex_sets,2):
            tmp=G.copy(immutable=False)
            tmp.add_vertex(k-2) # The vertices of graphs in F[k-2] have labels 0,...,k-3 due to storing the graphs with canonical labelling
            tmp.add_vertex(k-1)
            tmp.add_edge(k-2,k-1)
            for i in S:
                tmp.add_edge(i,k-2)
            for i in T:
                tmp.add_edge(i,k-1)
            edge_pivot(tmp,k-2,k-1)
            if not contains_forb_graph(tmp):
                F[k].add(tmp.canonical_label().copy(immutable=True))

for i in range(len(F)):
    print ("Number of minimal forbidden induced subgraphs on "+ str(i)+ " vertices: "+ str(len(F[i])))

if show_graph6:
    print "\nMinimal forbidden induced subgraphs in graph6 format:"
    for i in F:
        for j in i:
            print j.graph6_string()

if draw_graphs:
    print "\nShowing minimal forbidden induced subgraphs:"
    for i in F:
        for j in i:
            j.show()
