classdef RelaxationProblem < handle
    %RelaxationProblem class defining a relaxation problem
    
    properties (SetAccess=immutable)
        problemtype
        jumptype
        isQrelaxation
    end
    properties
       theta
       vS
       vH
       wCoupling
       verbose
       powderaverage
       alpha_RC
       beta_RC
       rotorsteps
       alphasteps
       betasteps
       colstr
       fitR1
    end
    properties (SetAccess = private)
        G
        W
        cons
        allgs
        allRs
        allTKs;
        updateW;
    end
    
    methods(Static)
        function initialisefit()
            global NEA NLOGTAU0 ND NTHETA R BOLTZ NA
            NEA=1; NLOGTAU0=2; ND=3; NTHETA=4;
            R=BOLTZ*NA;
        end
    end
    
    methods
        function obj= RelaxationProblem(problemtypev,jumptypev)
            if ~isa(problemtypev,'ProblemType')
                throw(MException('RelaxationProblem:noSuchModel', 'problemtype argument is not a valid model'));
            end
            if ~isa(jumptypev,'JumpType')
                throw(MException('RelaxationProblem:noSuchJumpModel', 'jumptype argument is not a valid jump model'));
            end
            
            switch jumptypev
                case JumpType.FREEROT,
                   disp('Model: free diffusion');
                case JumpType.THREEJUMP,
                    disp('Model: three site jump');
                case JumpType.TWOJUMP,
                    disp('Model: two site jump');
                otherwise
                    error('Unknown jump model');
            end
            obj.problemtype=problemtypev;
            switch obj.problemtype
                case {ProblemType.NDTWO,ProblemType.N2H}
                    obj.isQrelaxation=1;
                otherwise
                    obj.isQrelaxation=0;
            end
            obj.jumptype=jumptypev;
            obj.verbose=1;
            obj.powderaverage=1;
            obj.alpha_RC=30; % default angles for single orientation
            obj.beta_RC=30;
            obj.rotorsteps=30;
            obj.alphasteps=30;
            obj.betasteps=30;
            obj.colstr='rgbcmk';
            obj.fitR1=0;
        end
                
        function calculate(obj,alltauc,allTKs)
            Tsteps=length(alltauc);
            for tstep=1:Tsteps
                tauc=alltauc(tstep);
                D=1/tauc;
                if tstep==1
                    initialise(obj,tauc);
                end
                R=obj.calcR(tauc);
                %if obj.isQrelaxation~=0
                %    obj.G=calcQgs(D,obj.cons,obj.vH,obj.verbose);
                %else
                %    obj.G=calcgs(D,obj.cons,obj.vH,obj.vS,obj.verbose);
                %end
                if obj.verbose>0
                    TC=allTKs(tstep)-273.15;
                    fprintf('T=%g C   tau_c = %g x 10^-11 s   jump/diffusion rate: %g MHz\n',TC,tauc*1e11,D*1e-6);
                end
                if tstep==1
                    lallgs=zeros(Tsteps,length(obj.G));
                    lallRs=zeros(Tsteps,1);
                end
                lallgs(tstep,:)=obj.G;
                lallRs(tstep)=R;
            end
            % store result for plotting functions etc.
            obj.allTKs=allTKs;
            obj.allgs=lallgs;
            obj.allRs=lallRs;
        end
        
        function Rout= calcR(obj,tauc)
            D=1/tauc;
            if obj.isQrelaxation~=0
                obj.G=calcQgs(D,obj.cons,obj.vH,obj.verbose);
            else
                obj.G=calcgs(D,obj.cons,obj.vH,obj.vS,obj.verbose);
            end
            Rout=dot(obj.G,obj.W);
        end
        
        function plotresults(obj,plotfunc,nlabel,showcomponents)
            plotfunc(obj.allTKs-273.15, 1.0 ./ obj.allRs,'bx-');
            xlabel('T / C');
            ylabel([nlabel ' T_1 / s']);
            hold on;
            if showcomponents ~= 0
                for j=1:length(obj.W)
                    if obj.W(j)>0
        %semilogy(itscale,W(j).*allgs(:,j),[colstr(j),':']);
                        plotfunc(obj.allTKs-273.15,1.0 ./ (obj.W(j).*obj.allgs(:,j)),[obj.colstr(j),':']);
                    end
                end
            end
        end
        
        function [ fitparas, covar, chisqr, steps]= fitdata(obj,paras,tscale,ypoints,which,errors,noiselevel)
            parnames=cell(1,4);
            global NEA NLOGTAU0 ND NTHETA
            parnames{NEA}='Activation energy';
            parnames{NLOGTAU0}='log_10(tau_0)';
            parnames{ND}='Coupling / distance';
            parnames{NTHETA}='Cone angle';
            if any(which==NTHETA)
               obj.updateW=1;
               obj.W=[];
            else
               [obj.G,obj.W]=formrelax(obj,100e3);
               if obj.verbose>0
                   fprintf('Pre-calculating W factors\n');
               end
               obj.updateW=0;
            end
            oldverbose=obj.verbose;
            if obj.verbose>0
                obj.verbose=obj.verbose-1; % drop verbosity by one notch
            end
            [ fitparas, covar, chisqr, steps] =gofitdata( parnames,'relaxfunc', paras, tscale, ypoints, which, errors,noiselevel,50,0.2,0);
            obj.verbose=oldverbose;
        end
        
  % called by relaxfunc as constants etc updated
        function updateconstant(obj)
                obj.rawupdateconstant();
                if obj.updateW~=0
                    if obj.verbose>0
                        fprintf('Updating W factors\n');
                    end 
                    [obj.G,obj.W]=formrelax(obj,100e3);
                end
        end
        
        function initialise(obj,tauc)
            obj.rawupdateconstant();
            D=1/tauc;
            [obj.G,obj.W]=formrelax(obj,D);
        end
    end
    
    methods (Access= private)
        function rawupdateconstant(obj)
                switch obj.problemtype
                    case {ProblemType.NDTWO,ProblemType.N2H}
                        obj.cons=obj.wCoupling*obj.wCoupling/8;
                    case {ProblemType.N2HD,ProblemType.NME}
                      obj.cons=3*9*obj.wCoupling*obj.wCoupling/64; % factor of 3 because 3 x 1H!
                    case ProblemType.NCH
                        obj.cons=9*obj.wCoupling*obj.wCoupling/64; % without factor of 3!
                    otherwise
                        error('ProblemType not implemented!');
                end
                if obj.verbose>0
                    fprintf('Leading constant: %g rad^2 s^-2\n',obj.cons);
                end
        end

    end
    
end

