function [p,cov,chisqr,steps]=fitdata(func,p0,xpoints,ypoints,actord,gradv,origW,maxsteps,chistop,figno)
% FITDATA - fit y dataset to model function (Levenburg Marquardt)
%
%  [ p, cov, chisqr, steps] = fitdata(func,p0,xpoints,ypoints [,actord] [,errs] [,W] [,maxsteps] [,converg] [,figure])
%
%  Finds the values of the parameters (p) which best fit (in a least squares sense)
%  the model function which returns a y data set given the x values and parameters.
%
%  Algorithm: Levenburg Marquardt (adapted from Numerical Recipes)
%
%  Input parameters:
%	func: name of function (quoted) of the form ytrial=func(p,xpoints)
%	  p0: vector of starting parameters
%   xpoints: x values (fixed)
%   ypoints: y values to be fitted
%    actord: (optional) vector of parameters to be varied (by default all)
%      errs: (1e-2) vector of estimated error of each parameter or single scaling factor applied to all parameter values. Used for step size in numerical differentiation
%         W: (1.0) vector of noise standard deviation on each data point or single value applied to all. This allows chi^2 to be calculated appropriately.
%  maxsteps: (1000) Maximum number of iterations
%   converg: The change of chi^2 below which optimisation is assumed to have converged.
%     figno: (0) figure number to display fitting progress (0 for no display)
%
%  Output parameters:
%	  p: fitted parameters
%	cov: covariance matrix (for actord parameters)
%   chisqr: chi^2 at fitted point
%    steps: numbers of minimisation steps
%
rhs=nargin;

if rhs>10 || rhs<4
   error('Invalid number of arguments');
end

if size(p0,1)==1
	p0=p0.';
end

npars=length(p0);

npts=numel(ypoints);
%if prod(size(xpoints))~=npts
%   error('X and Y sets differ in length');
%end;

if rhs<5
	actord=[1:npars];
end

if rhs<6
	gradv=1e-2;
end
if length(gradv)==1
	gradv=gradv*p0;	
else
	if length(gradv)~=npars
		error('Parameter and error sets differ in length');
	end
end

W=origW;
if length(W)==1
   if W==0
      W=ones(1,npts);
   else
      W=W*ones(1,npts);
   end
elseif numel(W)~=npts
	error('Weighting and data sets differ in length');
end

if rhs<8
	maxsteps=1000;
end
if rhs<9
	chistop=npts*1e-5;
end
if maxsteps<1 || chistop<=0 
	error('Invalid convergence parameters');
end

if rhs<10
   figno=0;
end
origfig=gcf;

steps=0;

varpars=length(actord);

lambda=0.001;
disp('Starting point...');
[ochisqr,alpha,beta]=mqrcof(func,p0,xpoints,ypoints,actord,gradv,W,figno);
%    disp(alpha);
p=p0;

disp(['Initial chi2: ',num2str(ochisqr)]);
disp('Step  Parameters...  Chi^2');

while steps<maxsteps
	steps=steps+1;
	covar=alpha;
	for j=1:varpars
		covar(j,j)=alpha(j,j)*(1.0+lambda);
    end
	%da=inv(covar)*beta;
    da = covar \ beta;
	ptry=p;
	ptry(actord)=p(actord)+da;
	[chisqr,covar,da]=mqrcof(func,ptry,xpoints,ypoints,actord,gradv,W,figno);
    disp(num2str([ steps ptry.' chisqr ]));
	if chisqr<ochisqr
		lambda=lambda*0.1;
		alpha=covar;
		beta=da;
		p=ptry;
		if ochisqr-chisqr<chistop 
			break;
		end
		ochisqr=chisqr;
	else
		lambda=lambda*10.0;
	end
end;

if steps>=maxsteps
	disp('fitdata: max steps exceeded');
else
	disp('fitdata: converged');
end

if origW==0
    noiselevel=sqrt(chisqr/npts);
    fprintf('Using noiselevel of %g assuming residuals are random\n',noiselevel);
    W=noiselevel*ones(1,npts);
end
cov=covariance(func,p,xpoints,W,actord,gradv);

set(0,'CurrentFigure',origfig);
