# Copyright (C) 2015 Matthias C. M. Troffaes & Lewis Paton
# matthias.troffaes@durham.ac.uk, l.w.paton@durham.ac.uk
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Simple smoothed versions of the empirical transition probabilities."""

import numpy as np
from scipy.stats import multivariate_normal

def get_gaussian_kernel(bandwidth):
    """Gaussian kernel on univariate data."""

    mean = np.zeros(len(bandwidth))
    cov = np.diag(bandwidth ** 2)
    var = multivariate_normal(mean=mean, cov=cov)

    def pdf(x):
        assert len(x) == len(bandwidth)
        return var.pdf(x)

    return pdf

def get_smoother(xjs, kernel, j):

    def smoother(x):
        # matrix with kernel values in first column, and observations
        # of j in second column
        m = np.array([[kernel(x - xstar), int(j == jstar)]
                      for xstar, jstar in xjs])
        # kernel * observations / kernel
        return sum(m[:,0] * m[:,1]) / sum(m[:,0])

    return smoother

def get_density(xjs, kernel):

    def density(x):
        # matrix with kernel values in first column, and observations
        # of j in second column
        m = np.array([kernel(x - xstar) for xstar, jstar in xjs])
        # smoother denominator, renormalised against the kernel
        return sum(m) / kernel(np.zeros(len(x)))

    return density

